<?php
/**
 * PulseWatch Service Model
 */

class Service {
    private $db;

    public function __construct() {
        $this->db = db();
    }

    public function getAll($includeInactive = false) {
        $where = $includeInactive ? '1=1' : 'is_active = 1';
        return $this->db->fetchAll(
            "SELECT * FROM pw_services WHERE {$where} ORDER BY name ASC"
        );
    }

    public function getById($id) {
        return $this->db->fetch(
            "SELECT * FROM pw_services WHERE id = ?",
            [$id]
        );
    }

    public function create($data) {
        $service = [
            'name' => $data['name'],
            'url' => $data['url'],
            'type' => $data['type'] ?? 'https',
            'check_interval' => $data['check_interval'] ?? DEFAULT_CHECK_INTERVAL,
            'timeout' => $data['timeout'] ?? DEFAULT_TIMEOUT,
            'port' => $data['port'] ?? null,
            'keyword' => $data['keyword'] ?? null,
            'keyword_type' => $data['keyword_type'] ?? 'exists',
            'expected_status' => $data['expected_status'] ?? 200,
            'headers' => isset($data['headers']) ? json_encode($data['headers']) : null,
            'notify_email' => $data['notify_email'] ?? null,
            'notify_on_down' => $data['notify_on_down'] ?? 1,
            'notify_on_up' => $data['notify_on_up'] ?? 1,
            'fail_threshold' => $data['fail_threshold'] ?? FAIL_THRESHOLD,
            'is_active' => $data['is_active'] ?? 1,
            'status' => 'pending',
            'created_by' => $data['created_by'] ?? null
        ];

        return $this->db->insert('pw_services', $service);
    }

    public function update($id, $data) {
        $allowed = [
            'name', 'url', 'type', 'check_interval', 'timeout', 'port',
            'keyword', 'keyword_type', 'expected_status', 'headers',
            'notify_email', 'notify_on_down', 'notify_on_up', 
            'fail_threshold', 'is_active'
        ];

        $updateData = array_intersect_key($data, array_flip($allowed));
        
        if (isset($updateData['headers']) && is_array($updateData['headers'])) {
            $updateData['headers'] = json_encode($updateData['headers']);
        }

        if (empty($updateData)) {
            return false;
        }

        return $this->db->update('pw_services', $updateData, 'id = ?', [$id]);
    }

    public function delete($id) {
        return $this->db->delete('pw_services', 'id = ?', [$id]);
    }

    public function pause($id) {
        return $this->db->update('pw_services', 
            ['status' => 'paused', 'is_active' => 0], 
            'id = ?', 
            [$id]
        );
    }

    public function resume($id) {
        return $this->db->update('pw_services', 
            ['status' => 'pending', 'is_active' => 1], 
            'id = ?', 
            [$id]
        );
    }

    public function updateStatus($id, $status, $responseTime = null, $errorMessage = null) {
        $data = [
            'status' => $status,
            'last_check' => date('Y-m-d H:i:s'),
            'last_response_time' => $responseTime
        ];

        if ($status === 'down') {
            $this->db->query(
                "UPDATE pw_services SET consecutive_fails = consecutive_fails + 1 WHERE id = ?",
                [$id]
            );
        } else {
            $data['consecutive_fails'] = 0;
        }

        $this->db->update('pw_services', $data, 'id = ?', [$id]);

        // Log the check
        $this->logCheck($id, $status, $responseTime, null, $errorMessage);

        // Update uptime percentage
        $this->updateUptimePercentage($id);

        return true;
    }

    public function logCheck($serviceId, $status, $responseTime = null, $statusCode = null, $errorMessage = null) {
        return $this->db->insert('pw_checks', [
            'service_id' => $serviceId,
            'status' => $status,
            'response_time' => $responseTime,
            'status_code' => $statusCode,
            'error_message' => $errorMessage
        ]);
    }

    public function getChecks($serviceId, $limit = 100, $offset = 0) {
        return $this->db->fetchAll(
            "SELECT * FROM pw_checks WHERE service_id = ? ORDER BY checked_at DESC LIMIT ? OFFSET ?",
            [$serviceId, $limit, $offset]
        );
    }

    public function getRecentChecks($serviceId, $hours = 24) {
        return $this->db->fetchAll(
            "SELECT * FROM pw_checks WHERE service_id = ? AND checked_at >= DATE_SUB(NOW(), INTERVAL ? HOUR) ORDER BY checked_at ASC",
            [$serviceId, $hours]
        );
    }

    public function getUptimeHistory($serviceId, $days = 30) {
        return $this->db->fetchAll(
            "SELECT 
                DATE(checked_at) as date,
                COUNT(*) as total_checks,
                SUM(CASE WHEN status = 'up' THEN 1 ELSE 0 END) as up_checks,
                AVG(response_time) as avg_response_time,
                ROUND(SUM(CASE WHEN status = 'up' THEN 1 ELSE 0 END) / COUNT(*) * 100, 2) as uptime_percentage
            FROM pw_checks 
            WHERE service_id = ? AND checked_at >= DATE_SUB(CURDATE(), INTERVAL ? DAY)
            GROUP BY DATE(checked_at)
            ORDER BY date ASC",
            [$serviceId, $days]
        );
    }

    public function updateUptimePercentage($serviceId) {
        $result = $this->db->fetch(
            "SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'up' THEN 1 ELSE 0 END) as up_count
            FROM pw_checks 
            WHERE service_id = ? AND checked_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)",
            [$serviceId]
        );

        if ($result && $result['total'] > 0) {
            $uptime = round(($result['up_count'] / $result['total']) * 100, 2);
            $this->db->update('pw_services', ['uptime_percentage' => $uptime], 'id = ?', [$serviceId]);
        }
    }

    public function getServicesToCheck() {
        return $this->db->fetchAll(
            "SELECT * FROM pw_services 
            WHERE is_active = 1 
            AND (last_check IS NULL OR last_check <= DATE_SUB(NOW(), INTERVAL check_interval SECOND))
            ORDER BY last_check ASC
            LIMIT ?",
            [MAX_CONCURRENT_CHECKS]
        );
    }

    public function getStatusSummary() {
        return $this->db->fetch(
            "SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'up' THEN 1 ELSE 0 END) as up,
                SUM(CASE WHEN status = 'down' THEN 1 ELSE 0 END) as down,
                SUM(CASE WHEN status = 'paused' THEN 1 ELSE 0 END) as paused,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                AVG(uptime_percentage) as avg_uptime
            FROM pw_services WHERE is_active = 1"
        );
    }

    public function getOverallUptime($days = 30) {
        $result = $this->db->fetch(
            "SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'up' THEN 1 ELSE 0 END) as up_count
            FROM pw_checks 
            WHERE checked_at >= DATE_SUB(NOW(), INTERVAL ? DAY)",
            [$days]
        );

        if ($result && $result['total'] > 0) {
            return round(($result['up_count'] / $result['total']) * 100, 2);
        }
        return 100;
    }
}
