<?php
/**
 * PulseWatch Monitor - Performs service checks
 */

class Monitor {
    private $db;
    private $service;

    public function __construct() {
        $this->db = db();
        $this->service = new Service();
    }

    public function check($serviceData) {
        $startTime = microtime(true);
        $result = [
            'status' => 'down',
            'response_time' => null,
            'status_code' => null,
            'error' => null
        ];

        try {
            switch ($serviceData['type']) {
                case 'http':
                case 'https':
                    $result = $this->checkHttp($serviceData);
                    break;
                case 'ping':
                    $result = $this->checkPing($serviceData);
                    break;
                case 'port':
                    $result = $this->checkPort($serviceData);
                    break;
                case 'keyword':
                    $result = $this->checkKeyword($serviceData);
                    break;
                default:
                    $result = $this->checkHttp($serviceData);
            }
        } catch (Exception $e) {
            $result['error'] = $e->getMessage();
        }

        $result['response_time'] = round((microtime(true) - $startTime) * 1000);

        // Update service status
        $previousStatus = $serviceData['status'];
        $this->service->updateStatus(
            $serviceData['id'],
            $result['status'],
            $result['response_time'],
            $result['error']
        );

        // Handle status changes
        if ($previousStatus !== $result['status'] && $previousStatus !== 'pending') {
            $this->handleStatusChange($serviceData, $previousStatus, $result);
        }

        return $result;
    }

    private function checkHttp($service) {
        $ch = curl_init();
        
        $url = $service['url'];
        if (!preg_match('/^https?:\/\//', $url)) {
            $url = ($service['type'] === 'https' ? 'https://' : 'http://') . $url;
        }

        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => $service['timeout'] ?? DEFAULT_TIMEOUT,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS => 5,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
            CURLOPT_USERAGENT => 'PulseWatch Monitor/1.0',
            CURLOPT_NOBODY => false,
            CURLOPT_HEADER => false
        ]);

        // Add custom headers if specified
        if (!empty($service['headers'])) {
            $headers = json_decode($service['headers'], true);
            if ($headers) {
                $headerArray = [];
                foreach ($headers as $key => $value) {
                    $headerArray[] = "{$key}: {$value}";
                }
                curl_setopt($ch, CURLOPT_HTTPHEADER, $headerArray);
            }
        }

        $response = curl_exec($ch);
        $statusCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            return [
                'status' => 'down',
                'status_code' => $statusCode,
                'error' => $error
            ];
        }

        $expectedStatus = $service['expected_status'] ?? 200;
        $isUp = $statusCode >= 200 && $statusCode < 400;
        
        // Check for specific expected status
        if ($expectedStatus && $statusCode != $expectedStatus) {
            $isUp = false;
        }

        return [
            'status' => $isUp ? 'up' : 'down',
            'status_code' => $statusCode,
            'error' => $isUp ? null : "HTTP status {$statusCode}"
        ];
    }

    private function checkPing($service) {
        $host = parse_url($service['url'], PHP_URL_HOST) ?: $service['url'];
        $host = preg_replace('/[^a-zA-Z0-9\.\-]/', '', $host);
        
        // Use different ping command based on OS
        if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            $cmd = "ping -n 1 -w " . ($service['timeout'] * 1000) . " " . escapeshellarg($host);
        } else {
            $cmd = "ping -c 1 -W " . $service['timeout'] . " " . escapeshellarg($host) . " 2>&1";
        }

        exec($cmd, $output, $returnCode);
        $outputStr = implode("\n", $output);

        if ($returnCode === 0) {
            // Try to extract response time
            if (preg_match('/time[=<](\d+(?:\.\d+)?)\s*ms/i', $outputStr, $matches)) {
                return [
                    'status' => 'up',
                    'response_time' => (int) $matches[1],
                    'error' => null
                ];
            }
            return ['status' => 'up', 'error' => null];
        }

        return [
            'status' => 'down',
            'error' => 'Ping failed'
        ];
    }

    private function checkPort($service) {
        $host = parse_url($service['url'], PHP_URL_HOST) ?: $service['url'];
        $port = $service['port'] ?? 80;
        $timeout = $service['timeout'] ?? DEFAULT_TIMEOUT;

        $startTime = microtime(true);
        $connection = @fsockopen($host, $port, $errno, $errstr, $timeout);
        $responseTime = round((microtime(true) - $startTime) * 1000);

        if ($connection) {
            fclose($connection);
            return [
                'status' => 'up',
                'response_time' => $responseTime,
                'error' => null
            ];
        }

        return [
            'status' => 'down',
            'response_time' => $responseTime,
            'error' => "Port {$port}: {$errstr}"
        ];
    }

    private function checkKeyword($service) {
        $result = $this->checkHttp($service);
        
        if ($result['status'] === 'down') {
            return $result;
        }

        // Now check for keyword
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $service['url'],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => $service['timeout'] ?? DEFAULT_TIMEOUT,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_USERAGENT => 'PulseWatch Monitor/1.0'
        ]);

        $response = curl_exec($ch);
        curl_close($ch);

        $keyword = $service['keyword'];
        $keywordType = $service['keyword_type'] ?? 'exists';
        $keywordFound = stripos($response, $keyword) !== false;

        $isUp = ($keywordType === 'exists' && $keywordFound) || 
                ($keywordType === 'not_exists' && !$keywordFound);

        return [
            'status' => $isUp ? 'up' : 'down',
            'status_code' => $result['status_code'],
            'error' => $isUp ? null : "Keyword check failed: '{$keyword}' " . 
                      ($keywordType === 'exists' ? 'not found' : 'found')
        ];
    }

    private function handleStatusChange($service, $previousStatus, $result) {
        if ($result['status'] === 'down' && $previousStatus === 'up') {
            // Service went down
            $this->createIncident($service, $result['error']);
            
            // Check if we should notify (consecutive fails threshold)
            $serviceData = $this->service->getById($service['id']);
            if ($serviceData['consecutive_fails'] >= $serviceData['fail_threshold']) {
                $this->sendNotification($service, 'down', $result['error']);
            }
        } elseif ($result['status'] === 'up' && $previousStatus === 'down') {
            // Service came back up
            $this->resolveIncident($service['id']);
            $this->sendNotification($service, 'up');
        }
    }

    private function createIncident($service, $errorMessage = null) {
        return $this->db->insert('pw_incidents', [
            'service_id' => $service['id'],
            'error_message' => $errorMessage,
            'is_resolved' => 0
        ]);
    }

    private function resolveIncident($serviceId) {
        $incident = $this->db->fetch(
            "SELECT * FROM pw_incidents WHERE service_id = ? AND is_resolved = 0 ORDER BY started_at DESC LIMIT 1",
            [$serviceId]
        );

        if ($incident) {
            $duration = time() - strtotime($incident['started_at']);
            $this->db->update('pw_incidents', [
                'ended_at' => date('Y-m-d H:i:s'),
                'duration' => $duration,
                'is_resolved' => 1
            ], 'id = ?', [$incident['id']]);
        }
    }

    private function sendNotification($service, $type, $errorMessage = null) {
        $notifyEmail = $service['notify_email'];
        if (!$notifyEmail) {
            return;
        }

        // Check notification preferences
        if (($type === 'down' && !$service['notify_on_down']) ||
            ($type === 'up' && !$service['notify_on_up'])) {
            return;
        }

        $subject = $type === 'down' 
            ? "[PulseWatch] Alert: {$service['name']} is DOWN"
            : "[PulseWatch] Recovery: {$service['name']} is UP";

        $message = $type === 'down'
            ? "Your service '{$service['name']}' ({$service['url']}) is currently DOWN.\n\nError: {$errorMessage}\n\nTime: " . date('Y-m-d H:i:s')
            : "Your service '{$service['name']}' ({$service['url']}) has recovered and is now UP.\n\nTime: " . date('Y-m-d H:i:s');

        // Log the notification
        $this->db->insert('pw_notifications', [
            'service_id' => $service['id'],
            'type' => $type,
            'message' => $message,
            'sent_to' => $notifyEmail,
            'status' => 'sent'
        ]);

        // Send email
        $headers = [
            'From: PulseWatch <noreply@' . ($_SERVER['HTTP_HOST'] ?? 'localhost') . '>',
            'Content-Type: text/plain; charset=UTF-8'
        ];

        @mail($notifyEmail, $subject, $message, implode("\r\n", $headers));
    }

    public function runChecks() {
        $services = $this->service->getServicesToCheck();
        $results = [];

        foreach ($services as $serviceData) {
            $results[$serviceData['id']] = $this->check($serviceData);
        }

        return $results;
    }

    public function cleanupOldData($days = null) {
        if ($days === null) {
            $setting = $this->db->fetch(
                "SELECT setting_value FROM pw_settings WHERE setting_key = 'retention_days'"
            );
            $days = $setting ? (int) $setting['setting_value'] : 90;
        }

        // Delete old checks
        $this->db->query(
            "DELETE FROM pw_checks WHERE checked_at < DATE_SUB(NOW(), INTERVAL ? DAY)",
            [$days]
        );

        // Delete old resolved incidents
        $this->db->query(
            "DELETE FROM pw_incidents WHERE is_resolved = 1 AND ended_at < DATE_SUB(NOW(), INTERVAL ? DAY)",
            [$days]
        );

        // Delete old notifications
        $this->db->query(
            "DELETE FROM pw_notifications WHERE sent_at < DATE_SUB(NOW(), INTERVAL ? DAY)",
            [$days]
        );
    }
}
