<?php
/**
 * PulseWatch Incident Model
 */

class Incident {
    private $db;

    public function __construct() {
        $this->db = db();
    }

    public function getAll($limit = 50, $offset = 0) {
        return $this->db->fetchAll(
            "SELECT i.*, s.name as service_name, s.url as service_url
            FROM pw_incidents i
            JOIN pw_services s ON i.service_id = s.id
            ORDER BY i.started_at DESC
            LIMIT ? OFFSET ?",
            [$limit, $offset]
        );
    }

    public function getByService($serviceId, $limit = 50) {
        return $this->db->fetchAll(
            "SELECT * FROM pw_incidents 
            WHERE service_id = ? 
            ORDER BY started_at DESC 
            LIMIT ?",
            [$serviceId, $limit]
        );
    }

    public function getActive() {
        return $this->db->fetchAll(
            "SELECT i.*, s.name as service_name, s.url as service_url
            FROM pw_incidents i
            JOIN pw_services s ON i.service_id = s.id
            WHERE i.is_resolved = 0
            ORDER BY i.started_at DESC"
        );
    }

    public function getById($id) {
        return $this->db->fetch(
            "SELECT i.*, s.name as service_name, s.url as service_url
            FROM pw_incidents i
            JOIN pw_services s ON i.service_id = s.id
            WHERE i.id = ?",
            [$id]
        );
    }

    public function resolve($id) {
        $incident = $this->getById($id);
        if (!$incident || $incident['is_resolved']) {
            return false;
        }

        $duration = time() - strtotime($incident['started_at']);
        return $this->db->update('pw_incidents', [
            'ended_at' => date('Y-m-d H:i:s'),
            'duration' => $duration,
            'is_resolved' => 1
        ], 'id = ?', [$id]);
    }

    public function getStats($days = 30) {
        return $this->db->fetch(
            "SELECT 
                COUNT(*) as total_incidents,
                SUM(CASE WHEN is_resolved = 0 THEN 1 ELSE 0 END) as active_incidents,
                AVG(duration) as avg_duration,
                MAX(duration) as max_duration
            FROM pw_incidents 
            WHERE started_at >= DATE_SUB(NOW(), INTERVAL ? DAY)",
            [$days]
        );
    }

    public function getIncidentsByDay($days = 30) {
        return $this->db->fetchAll(
            "SELECT 
                DATE(started_at) as date,
                COUNT(*) as count
            FROM pw_incidents 
            WHERE started_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
            GROUP BY DATE(started_at)
            ORDER BY date ASC",
            [$days]
        );
    }

    public function getTotalDowntime($serviceId = null, $days = 30) {
        $params = [$days];
        $where = "started_at >= DATE_SUB(NOW(), INTERVAL ? DAY)";
        
        if ($serviceId) {
            $where .= " AND service_id = ?";
            $params[] = $serviceId;
        }

        $result = $this->db->fetch(
            "SELECT SUM(duration) as total_downtime FROM pw_incidents WHERE {$where} AND is_resolved = 1",
            $params
        );

        return $result ? (int) $result['total_downtime'] : 0;
    }
}
