<?php
/**
 * PulseWatch Authentication Handler
 */

class Auth {
    private $db;

    public function __construct() {
        $this->db = db();
    }

    public function login($username, $password) {
        $user = $this->db->fetch(
            "SELECT * FROM pw_users WHERE username = ? OR email = ?",
            [$username, $username]
        );

        if ($user && password_verify($password, $user['password'])) {
            $this->setSession($user);
            return [
                'success' => true,
                'user' => $this->sanitizeUser($user)
            ];
        }

        return ['success' => false, 'error' => 'Invalid credentials'];
    }

    public function logout() {
        $_SESSION = [];
        if (isset($_COOKIE[session_name()])) {
            setcookie(session_name(), '', time() - 3600, '/');
        }
        session_destroy();
    }

    public function isLoggedIn() {
        return isset($_SESSION['user_id']) && $this->validateSession();
    }

    public function isAdmin() {
        return $this->isLoggedIn() && $_SESSION['user_role'] === 'admin';
    }

    public function getCurrentUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        return $this->db->fetch("SELECT * FROM pw_users WHERE id = ?", [$_SESSION['user_id']]);
    }

    public function requireAuth() {
        if (!$this->isLoggedIn()) {
            http_response_code(401);
            echo json_encode(['error' => 'Authentication required']);
            exit;
        }
    }

    public function requireAdmin() {
        $this->requireAuth();
        if (!$this->isAdmin()) {
            http_response_code(403);
            echo json_encode(['error' => 'Admin access required']);
            exit;
        }
    }

    public function createUser($data) {
        if ($this->db->count('pw_users', 'username = ? OR email = ?', [$data['username'], $data['email']]) > 0) {
            return ['success' => false, 'error' => 'Username or email already exists'];
        }

        $id = $this->db->insert('pw_users', [
            'username' => $data['username'],
            'email' => $data['email'],
            'password' => password_hash($data['password'], PASSWORD_DEFAULT),
            'role' => $data['role'] ?? 'viewer'
        ]);

        return ['success' => true, 'id' => $id];
    }

    public function updatePassword($userId, $currentPassword, $newPassword) {
        $user = $this->db->fetch("SELECT password FROM pw_users WHERE id = ?", [$userId]);
        
        if (!$user || !password_verify($currentPassword, $user['password'])) {
            return ['success' => false, 'error' => 'Current password is incorrect'];
        }

        $this->db->update('pw_users', 
            ['password' => password_hash($newPassword, PASSWORD_DEFAULT)],
            'id = ?',
            [$userId]
        );

        return ['success' => true];
    }

    public function generateApiToken($userId) {
        $token = bin2hex(random_bytes(32));
        $hashedToken = hash('sha256', $token);
        
        // Store hashed token (you'd need an api_tokens table for this)
        // For simplicity, we'll use session-based auth in this version
        
        return $token;
    }

    public function validateApiToken($token) {
        $hashedToken = hash('sha256', $token);
        // Validate against stored tokens
        // For now, return false as we're using session auth
        return false;
    }

    private function setSession($user) {
        session_regenerate_id(true);
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['user_role'] = $user['role'];
        $_SESSION['user_name'] = $user['username'];
        $_SESSION['login_time'] = time();
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }

    private function validateSession() {
        if (!isset($_SESSION['login_time'])) {
            return false;
        }
        
        if (time() - $_SESSION['login_time'] > SESSION_LIFETIME) {
            $this->logout();
            return false;
        }
        
        return true;
    }

    private function sanitizeUser($user) {
        unset($user['password']);
        return $user;
    }

    public function getCsrfToken() {
        if (!isset($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }

    public function validateCsrf($token) {
        return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
    }
}
