#!/usr/bin/php
<?php
/**
 * PulseWatch Cron Monitor
 * 
 * Run this script via cron every minute:
 * * * * * * /usr/bin/php /path/to/pulsewatch/cron/monitor.php >> /path/to/pulsewatch/logs/cron.log 2>&1
 */

// Prevent web access
if (php_sapi_name() !== 'cli') {
    die('This script can only be run from the command line.');
}

// Set working directory
chdir(dirname(__DIR__));

// Load configuration
require_once __DIR__ . '/../config/config.php';
require_once INCLUDES_PATH . '/Database.php';
require_once INCLUDES_PATH . '/Auth.php';
require_once INCLUDES_PATH . '/Service.php';
require_once INCLUDES_PATH . '/Monitor.php';
require_once INCLUDES_PATH . '/Incident.php';

// Check if monitoring is enabled
$db = db();
$setting = $db->fetch("SELECT setting_value FROM pw_settings WHERE setting_key = 'check_enabled'");
if (!$setting || $setting['setting_value'] != '1') {
    echo date('Y-m-d H:i:s') . " - Monitoring is disabled\n";
    exit(0);
}

// Prevent overlapping runs using a lock file
$lockFile = LOGS_PATH . '/monitor.lock';
$fp = fopen($lockFile, 'w');

if (!flock($fp, LOCK_EX | LOCK_NB)) {
    echo date('Y-m-d H:i:s') . " - Another instance is already running\n";
    exit(0);
}

// Write PID to lock file
fwrite($fp, getmypid());

try {
    echo date('Y-m-d H:i:s') . " - Starting monitoring run\n";
    
    $monitor = new Monitor();
    $results = $monitor->runChecks();
    
    $upCount = 0;
    $downCount = 0;
    
    foreach ($results as $serviceId => $result) {
        if ($result['status'] === 'up') {
            $upCount++;
        } else {
            $downCount++;
        }
        
        echo sprintf(
            "  Service #%d: %s (response: %dms)\n",
            $serviceId,
            strtoupper($result['status']),
            $result['response_time'] ?? 0
        );
        
        if ($result['error']) {
            echo "    Error: {$result['error']}\n";
        }
    }
    
    echo date('Y-m-d H:i:s') . " - Completed: {$upCount} up, {$downCount} down\n";
    
    // Cleanup old data (run once daily - check hour)
    if (date('G') === '3' && date('i') < 5) {
        echo date('Y-m-d H:i:s') . " - Running cleanup\n";
        $monitor->cleanupOldData();
    }
    
} catch (Exception $e) {
    echo date('Y-m-d H:i:s') . " - Error: " . $e->getMessage() . "\n";
    error_log("Cron Monitor Error: " . $e->getMessage());
} finally {
    // Release lock
    flock($fp, LOCK_UN);
    fclose($fp);
    @unlink($lockFile);
}
