<?php
/**
 * PulseWatch API Router
 */

require_once __DIR__ . '/../config/config.php';
require_once INCLUDES_PATH . '/Database.php';
require_once INCLUDES_PATH . '/Auth.php';
require_once INCLUDES_PATH . '/Service.php';
require_once INCLUDES_PATH . '/Monitor.php';
require_once INCLUDES_PATH . '/Incident.php';

// CORS Headers
header('Access-Control-Allow-Origin: ' . CORS_ORIGIN);
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-CSRF-Token');
header('Access-Control-Allow-Credentials: true');
header('Content-Type: application/json');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Initialize
$auth = new Auth();
$service = new Service();
$incident = new Incident();

// Parse request
$method = $_SERVER['REQUEST_METHOD'];
$uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$uri = preg_replace('/^.*\/api\//', '', $uri);
$parts = explode('/', trim($uri, '/'));
$endpoint = $parts[0] ?? '';
$id = $parts[1] ?? null;
$action = $parts[2] ?? null;

// Get request body
$input = json_decode(file_get_contents('php://input'), true) ?? [];

// Response helper
function jsonResponse($data, $code = 200) {
    http_response_code($code);
    echo json_encode($data);
    exit;
}

function errorResponse($message, $code = 400) {
    jsonResponse(['error' => $message], $code);
}

// Routes
try {
    switch ($endpoint) {
        // ==================
        // AUTH ENDPOINTS
        // ==================
        case 'auth':
            switch ($action ?? $id) {
                case 'login':
                    if ($method !== 'POST') errorResponse('Method not allowed', 405);
                    if (empty($input['username']) || empty($input['password'])) {
                        errorResponse('Username and password required');
                    }
                    $result = $auth->login($input['username'], $input['password']);
                    if ($result['success']) {
                        jsonResponse([
                            'success' => true,
                            'user' => $result['user'],
                            'csrf_token' => $auth->getCsrfToken()
                        ]);
                    } else {
                        errorResponse($result['error'], 401);
                    }
                    break;

                case 'logout':
                    $auth->logout();
                    jsonResponse(['success' => true]);
                    break;

                case 'check':
                    jsonResponse([
                        'authenticated' => $auth->isLoggedIn(),
                        'isAdmin' => $auth->isAdmin(),
                        'user' => $auth->isLoggedIn() ? $auth->getCurrentUser() : null,
                        'csrf_token' => $auth->getCsrfToken()
                    ]);
                    break;

                default:
                    errorResponse('Unknown auth action', 404);
            }
            break;

        // ==================
        // SERVICES ENDPOINTS
        // ==================
        case 'services':
            if ($id === null) {
                // GET /services - List all services
                if ($method === 'GET') {
                    $includeInactive = isset($_GET['all']) && $auth->isAdmin();
                    jsonResponse($service->getAll($includeInactive));
                }
                // POST /services - Create service (admin only)
                elseif ($method === 'POST') {
                    $auth->requireAdmin();
                    if (empty($input['name']) || empty($input['url'])) {
                        errorResponse('Name and URL required');
                    }
                    $input['created_by'] = $_SESSION['user_id'];
                    $newId = $service->create($input);
                    jsonResponse(['success' => true, 'id' => $newId], 201);
                }
                else {
                    errorResponse('Method not allowed', 405);
                }
            } else {
                // Actions on specific service
                if ($action === 'pause') {
                    $auth->requireAdmin();
                    $service->pause($id);
                    jsonResponse(['success' => true]);
                }
                elseif ($action === 'resume') {
                    $auth->requireAdmin();
                    $service->resume($id);
                    jsonResponse(['success' => true]);
                }
                elseif ($action === 'checks') {
                    $limit = $_GET['limit'] ?? 100;
                    $hours = $_GET['hours'] ?? 24;
                    jsonResponse($service->getRecentChecks($id, $hours));
                }
                elseif ($action === 'history') {
                    $days = $_GET['days'] ?? 30;
                    jsonResponse($service->getUptimeHistory($id, $days));
                }
                elseif ($action === 'incidents') {
                    jsonResponse($incident->getByService($id));
                }
                else {
                    // GET /services/:id
                    if ($method === 'GET') {
                        $data = $service->getById($id);
                        if (!$data) errorResponse('Service not found', 404);
                        jsonResponse($data);
                    }
                    // PUT /services/:id
                    elseif ($method === 'PUT') {
                        $auth->requireAdmin();
                        $service->update($id, $input);
                        jsonResponse(['success' => true]);
                    }
                    // DELETE /services/:id
                    elseif ($method === 'DELETE') {
                        $auth->requireAdmin();
                        $service->delete($id);
                        jsonResponse(['success' => true]);
                    }
                    else {
                        errorResponse('Method not allowed', 405);
                    }
                }
            }
            break;

        // ==================
        // INCIDENTS ENDPOINTS
        // ==================
        case 'incidents':
            if ($id === null) {
                // GET /incidents
                if ($method === 'GET') {
                    $active = isset($_GET['active']);
                    if ($active) {
                        jsonResponse($incident->getActive());
                    } else {
                        jsonResponse($incident->getAll());
                    }
                }
                else {
                    errorResponse('Method not allowed', 405);
                }
            } else {
                if ($action === 'resolve') {
                    $auth->requireAdmin();
                    $incident->resolve($id);
                    jsonResponse(['success' => true]);
                } else {
                    // GET /incidents/:id
                    if ($method === 'GET') {
                        $data = $incident->getById($id);
                        if (!$data) errorResponse('Incident not found', 404);
                        jsonResponse($data);
                    }
                    else {
                        errorResponse('Method not allowed', 405);
                    }
                }
            }
            break;

        // ==================
        // DASHBOARD/STATS ENDPOINTS
        // ==================
        case 'dashboard':
        case 'stats':
            $days = $_GET['days'] ?? 30;
            
            $summary = $service->getStatusSummary();
            $overallUptime = $service->getOverallUptime($days);
            $incidentStats = $incident->getStats($days);
            $activeIncidents = $incident->getActive();

            jsonResponse([
                'summary' => $summary,
                'overall_uptime' => $overallUptime,
                'incident_stats' => $incidentStats,
                'active_incidents' => $activeIncidents,
                'services' => $service->getAll()
            ]);
            break;

        // ==================
        // MONITORING TRIGGER (for cron)
        // ==================
        case 'monitor':
            if ($action === 'run') {
                // Optional: Add security token check for cron
                $monitor = new Monitor();
                $results = $monitor->runChecks();
                jsonResponse(['success' => true, 'checked' => count($results), 'results' => $results]);
            }
            elseif ($action === 'cleanup') {
                $auth->requireAdmin();
                $monitor = new Monitor();
                $monitor->cleanupOldData();
                jsonResponse(['success' => true]);
            }
            else {
                errorResponse('Unknown monitor action', 404);
            }
            break;

        // ==================
        // SETTINGS (admin only)
        // ==================
        case 'settings':
            $auth->requireAdmin();
            
            if ($method === 'GET') {
                $settings = db()->fetchAll("SELECT * FROM pw_settings");
                $result = [];
                foreach ($settings as $s) {
                    $result[$s['setting_key']] = $s['setting_value'];
                }
                jsonResponse($result);
            }
            elseif ($method === 'PUT') {
                foreach ($input as $key => $value) {
                    db()->query(
                        "UPDATE pw_settings SET setting_value = ? WHERE setting_key = ?",
                        [$value, $key]
                    );
                }
                jsonResponse(['success' => true]);
            }
            else {
                errorResponse('Method not allowed', 405);
            }
            break;

        // ==================
        // USERS (admin only)
        // ==================
        case 'users':
            $auth->requireAdmin();
            
            if ($id === null) {
                if ($method === 'GET') {
                    $users = db()->fetchAll("SELECT id, username, email, role, created_at FROM pw_users");
                    jsonResponse($users);
                }
                elseif ($method === 'POST') {
                    if (empty($input['username']) || empty($input['email']) || empty($input['password'])) {
                        errorResponse('Username, email and password required');
                    }
                    $result = $auth->createUser($input);
                    if ($result['success']) {
                        jsonResponse(['success' => true, 'id' => $result['id']], 201);
                    } else {
                        errorResponse($result['error']);
                    }
                }
                else {
                    errorResponse('Method not allowed', 405);
                }
            } else {
                if ($method === 'DELETE') {
                    if ($id == $_SESSION['user_id']) {
                        errorResponse('Cannot delete yourself');
                    }
                    db()->delete('pw_users', 'id = ?', [$id]);
                    jsonResponse(['success' => true]);
                }
                elseif ($method === 'PUT') {
                    $allowed = ['username', 'email', 'role'];
                    $updateData = array_intersect_key($input, array_flip($allowed));
                    if (!empty($input['password'])) {
                        $updateData['password'] = password_hash($input['password'], PASSWORD_DEFAULT);
                    }
                    db()->update('pw_users', $updateData, 'id = ?', [$id]);
                    jsonResponse(['success' => true]);
                }
                else {
                    errorResponse('Method not allowed', 405);
                }
            }
            break;

        default:
            errorResponse('Endpoint not found', 404);
    }
} catch (Exception $e) {
    error_log("API Error: " . $e->getMessage());
    errorResponse('Internal server error', 500);
}
